<?php
/**
 * Lista TODOS os SKUs e seus estoques diretamente da Nuvemshop.
 * - Pagina /v1/{store}/products até acabar (404 "Last page is X" = fim)
 * - Para cada produto, lê as variantes e calcula estoque:
 *     * Se houver inventory_levels: soma os stocks de todos os depósitos
 *     * Senão, usa o campo legacy variant.stock (quando presente)
 * - Saída: JSON (default) ou CSV (?csv=1)
 */

require_once "../_class/Conexao_class.php";
require_once "../_class/Crud_class.php";
require_once "../_class/Emitente_class.php";
require_once "../_class/parametros.php";
require_once "../../functions.php";

date_default_timezone_set("America/Sao_Paulo");
ini_set('memory_limit', '-1');
set_time_limit(0);

date_default_timezone_set("America/Sao_Paulo");
ini_set('memory_limit', '-1');
set_time_limit(0);

// ===== Credenciais =====
$user_id      = $emitente->user_id  ?? null;
$access_token = $emitente->access_token ?? null;
if (!$user_id || !$access_token) {
    http_response_code(500);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(["retorno"=>false,"erro"=>"Credenciais Nuvemshop ausentes (user_id/access_token)."], JSON_UNESCAPED_UNICODE);
    exit;
}
$AUTH = "Authentication: bearer " . $access_token;
$UA   = "T2 Sistemas (contato@t2sistemas.com)";

// ===== Parâmetros =====
$PER_PAGE = isset($_GET['per_page']) ? max(1, min(200, (int)$_GET['per_page'])) : 200; // API costuma aceitar até 200
$INCLUDE_EMPTY_SKU = isset($_GET['include_empty_sku']) ? (bool)(int)$_GET['include_empty_sku'] : false; // ?include_empty_sku=1 para incluir variantes sem SKU

// ===== Helpers =====
function api_get_json(string $url, string $authHeader, string $ua): array {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL            => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HEADER         => false,
        CURLOPT_HTTPHEADER     => [
            $authHeader,
            'Content-Type: application/json',
            'User-Agent: ' . $ua
        ],
        CURLOPT_TIMEOUT        => 60,
    ]);
    $res  = curl_exec($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $err  = curl_error($ch);
    curl_close($ch);

    return [
        'http' => (int)$http,
        'err'  => $err ?: null,
        'raw'  => $res,
        'data' => $res !== false ? json_decode($res, true) : null,
    ];
}

/**
 * Extrai (sku, estoque) a partir do array de variants de um produto retornado em /products
 * (muitas lojas já recebem inventory_levels nesse payload).
 */
function variants_sku_and_stock_from_product_payload(array $product): array {
    $out = [];
    if (empty($product['variants']) || !is_array($product['variants'])) return $out;

    foreach ($product['variants'] as $v) {
        $sku = isset($v['sku']) ? trim((string)$v['sku']) : '';
        // Calcula estoque: soma inventory_levels[].stock; fallback em variant.stock
        $estoque = null;
        if (!empty($v['inventory_levels']) && is_array($v['inventory_levels'])) {
            $soma = 0;
            foreach ($v['inventory_levels'] as $lvl) {
                if (isset($lvl['stock'])) $soma += (int)$lvl['stock'];
            }
            $estoque = $soma;
        } elseif (array_key_exists('stock', $v)) {
            $estoque = (int)$v['stock'];
        }
        $out[] = ['sku' => $sku, 'estoque' => $estoque];
    }
    return $out;
}

// ===== Coleta de todos os SKUs =====
$linhas = [];
$page = 1;

while (true) {
    $url = "https://api.tiendanube.com/v1/{$user_id}/products?page={$page}&per_page={$PER_PAGE}";
    $resp = api_get_json($url, $AUTH, $UA);

    if ($resp['http'] === 404) {
        // "Last page is X" -> acabou
        break;
    }
    if ($resp['http'] >= 400) {
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode([
            "retorno"=>false,
            "erro"=>"Falha ao buscar products",
            "detalhe"=>"GET $url falhou (HTTP {$resp['http']}): {$resp['err']} | {$resp['raw']}"
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }
    if (!is_array($resp['data']) || count($resp['data']) === 0) {
        break;
    }

    foreach ($resp['data'] as $product) {
        $pairs = variants_sku_and_stock_from_product_payload($product);
        foreach ($pairs as $p) {
            $sku = $p['sku'];
            $stock = $p['estoque'];
            if ($sku === '' && !$INCLUDE_EMPTY_SKU) continue; // pula variantes sem SKU, a não ser que peça para incluir
            // Se quiser converter null para 0, descomente a linha abaixo:
            // if ($stock === null) $stock = 0;
            $linhas[] = ["sku"=>$sku, "estoque"=>$stock];
        }
    }

    $page++;
}

// Ordena por SKU
usort($linhas, function($a, $b) {
    return strcmp((string)$a['sku'], (string)$b['sku']);
});

// ===== Saída CSV ou JSON =====
if (isset($_GET['csv']) && (int)$_GET['csv'] === 1) {
    $filename = "nuvemshop_skus_estoques_" . date('Ymd_His') . ".csv";
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="'.$filename.'"');
    echo "\xEF\xBB\xBF"; // BOM p/ Excel
    $out = fopen('php://output', 'w');
    fputcsv($out, ['SKU','Estoque'], ';');
    foreach ($linhas as $linha) {
        fputcsv($out, [$linha['sku'], $linha['estoque']], ';');
    }
    fclose($out);
    exit;
}

header('Content-Type: application/json; charset=utf-8');
echo json_encode([
    "retorno"      => true,
    "total_linhas" => count($linhas),
    "linhas"       => $linhas
], JSON_UNESCAPED_UNICODE);
