<?php
/**
 * Atualização de estoque em massa (paginada) no estilo do seu script,
 * mas enviando em LOTE para a Nuvemshop.
 *
 * Paginação: ?page=1&per_page=500 (defaults: 1 e 500)
 * Para cada produto (limitado pela paginação), busca as variantes em estoqueprodutos
 * e envia em PATCH /v1/{user_id}/products/stock-price, fatiando em sub-lotes de 50 variantes.
 */

require_once "../_class/Conexao_class.php";
require_once "../_class/Crud_class.php";
require_once "../_class/Emitente_class.php";
require_once "../_class/parametros.php";
require_once "../../functions.php";

date_default_timezone_set("America/Sao_Paulo");
ini_set('memory_limit', '-1');
set_time_limit(0);

// ========= DEBUG OPCIONAL =========
// header('Content-Type: application/json; charset=utf-8');
// ini_set('display_errors', '1');
// error_reporting(E_ALL);
// ==================================

$pdo  = Conexao::getInstance();
$user_id      = $emitente->user_id ?? null;
$access_token = $emitente->access_token ?? null;

if (!$user_id || !$access_token) {
    http_response_code(500);
    echo json_encode(["retorno" => false, "erro" => "Credenciais Nuvemshop ausentes (user_id/access_token)."], JSON_UNESCAPED_UNICODE);
    exit;
}

$authHeader = "Authentication: bearer " . $access_token;
$userAgent  = "T2 Sistemas (contato@t2sistemas.com)";

// ======== Parâmetros de paginação ========
$page     = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$per_page = isset($_GET['per_page']) ? max(1, (int)$_GET['per_page']) : 500;
$offset   = ($page - 1) * $per_page;

// ======== Busca os produtos desta página ========
$crud = Crud::getInstance($pdo, 'produtos');
$sqlProdutos = "
    SELECT codigo, id_ecommerce
    FROM produtos
    WHERE ecommerce = 'S'
      AND id_ecommerce <> ''
    ORDER BY codigo
    LIMIT :offset, :limit
";
$stmt = $pdo->prepare($sqlProdutos);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->bindValue(':limit',  $per_page, PDO::PARAM_INT);
$stmt->execute();
$produtos = $stmt->fetchAll(PDO::FETCH_OBJ);

// Nada para processar
if (!$produtos || count($produtos) === 0) {
    echo json_encode([
        "retorno" => true,
        "msg"     => "Nenhum produto encontrado para a página especificada.",
        "page"    => $page,
        "per_page"=> $per_page,
        "total_variantes_encontradas" => 0,
        "batches_enviados" => 0
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// ======== Funções auxiliares ========

/** Envia um sub-lote (até 50 variantes) para o endpoint bulk da Nuvemshop. */
function enviarSubLoteBulk(int $user_id, string $authHeader, string $userAgent, array $payload) {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL            => "https://api.tiendanube.com/v1/{$user_id}/products/stock-price",
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HEADER         => false,
        CURLOPT_CUSTOMREQUEST  => 'PATCH',
        CURLOPT_POSTFIELDS     => json_encode($payload, JSON_UNESCAPED_UNICODE),
        CURLOPT_HTTPHEADER     => [
            $authHeader,
            'Content-Type: application/json',
            'User-Agent: ' . $userAgent
        ],
        CURLOPT_TIMEOUT        => 60,
    ]);
    $response = curl_exec($ch);
    $http     = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $err      = curl_error($ch);
    curl_close($ch);

    return [
        "http"     => $http,
        "erro"     => $err ?: null,
        "response" => $response
    ];
}

/** Adiciona uma variante ao payload (agrupa por product_id). */
function pushVariantToPayload(array &$payload, int $productId, int $variantId, int $stock) {
    $found = null;
    foreach ($payload as $i => $p) {
        if ($p['id'] === $productId) { $found = $i; break; }
    }
    if ($found === null) {
        $payload[] = [ "id" => $productId, "variants" => [] ];
        $found = count($payload) - 1;
    }
    // single-depósito: apenas "stock"
    $payload[$found]['variants'][] = [
        "id" => $variantId,
        "inventory_levels" => [
            [ "stock" => (int)$stock ]
        ]
    ];
}

// ======== Monta e envia em sub-lotes de 50 variantes ========
$SUBLOTE_MAX_VARIANTS = 50; // hard limit da API
$payloadSubLote = [];       // array de produtos com suas variants
$contadorVariantesNoSubLote = 0;

$totalProdutosProcessados  = 0;
$totalVariantesEncontradas = 0;
$batchesEnviados           = 0;
$detalhesEnvio             = []; // opcional: log por sub-lote
$erros                     = [];

foreach ($produtos as $p) {
    $totalProdutosProcessados++;

    // Para cada produto, busca as variantes no estoque
    $sqlVar = "
        SELECT 
            produto, codproduto, id_ecommerce, 
            GREATEST(COALESCE(qtd4,0),0) AS estoque
        FROM estoqueprodutos
        WHERE produto = :codigo
          AND id_ecommerce <> ''
    ";
    $stVar = $pdo->prepare($sqlVar);
    $stVar->bindValue(':codigo', $p->codigo);
    $stVar->execute();
    $itens = $stVar->fetchAll(PDO::FETCH_OBJ);

    if (!$itens) { continue; }

    foreach ($itens as $it) {
        $totalVariantesEncontradas++;

        // Empilha variante no payload do sub-lote (agrupado por produto)
        pushVariantToPayload($payloadSubLote, (int)$p->id_ecommerce, (int)$it->id_ecommerce, (int)$it->estoque);
        $contadorVariantesNoSubLote++;

        // Se batemos 50 variantes, dispara este sub-lote
        if ($contadorVariantesNoSubLote >= $SUBLOTE_MAX_VARIANTS) {
            $ret = enviarSubLoteBulk((int)$user_id, $authHeader, $userAgent, $payloadSubLote);
            $batchesEnviados++;

            if ($ret['http'] >= 400 || $ret['response'] === false) {
                $erros[] = [
                    "http" => $ret['http'],
                    "erro" => $ret['erro'],
                    "response" => $ret['response']
                ];
            } else {
                // opcional: guardar um resumo do lote
                $detalhesEnvio[] = [
                    "http" => $ret['http'],
                    "response_sample" => mb_substr((string)$ret['response'], 0, 300, 'UTF-8')
                ];
            }

            // reseta sub-lote
            $payloadSubLote = [];
            $contadorVariantesNoSubLote = 0;
        }
    }
}

// Envia o último sub-lote pendente (se houver)
if ($contadorVariantesNoSubLote > 0 && !empty($payloadSubLote)) {
    $ret = enviarSubLoteBulk((int)$user_id, $authHeader, $userAgent, $payloadSubLote);
    $batchesEnviados++;

    if ($ret['http'] >= 400 || $ret['response'] === false) {
        $erros[] = [
            "http" => $ret['http'],
            "erro" => $ret['erro'],
            "response" => $ret['response']
        ];
    } else {
        $detalhesEnvio[] = [
            "http" => $ret['http'],
            "response_sample" => mb_substr((string)$ret['response'], 0, 300, 'UTF-8')
        ];
    }
}

// Saída
echo json_encode([
    "retorno"                      => empty($erros),
    "page"                         => $page,
    "per_page"                     => $per_page,
    "offset"                       => $offset,
    "total_produtos_processados"   => $totalProdutosProcessados,
    "total_variantes_encontradas"  => $totalVariantesEncontradas,
    "batches_enviados"             => $batchesEnviados,
    "erros"                        => $erros,
    // "detalhes_envio"            => $detalhesEnvio, // habilite se quiser ver amostra da resposta dos lotes
], JSON_UNESCAPED_UNICODE);


echo "<script>
        setTimeout(function(){
            window.location.href = 'atualiza_estoque_massa.php?page=" . ($page+1) . "&per_page=$per_page';
        }, 2000);
      </script>";